//
//  MedicalPackage.m
//  as350b3e
//
//  Created by Vadim Dagman on 4/3/14.
//
//

#import "MedicalPackage.h"

@implementation MedicalItem

@synthesize weight=_weight;
@synthesize arm=_arm;
@synthesize armLat=_armLat;

- (id)initWithDictionary:(NSDictionary *)dict {
    self = [super init];
    if (self) {
        _station = [dict valueForKey:@"station"];
        _weight = [[dict valueForKey:@"weight"] floatValue];
        _selected = [[dict valueForKey:@"selected"] boolValue];
        _arm = [[dict valueForKey:@"arm"] floatValue];
        _armLat = [[dict valueForKey:@"armLat"] floatValue];
    }
    return self;
}

- (NSDictionary*)convertToDictionary {
    NSMutableDictionary* dict = [NSMutableDictionary dictionary];
    [dict setValue:_station forKey:@"station"];
    [dict setValue:[NSNumber numberWithFloat:_weight] forKey:@"weight"];
    [dict setValue:[NSNumber numberWithBool:_selected] forKey:@"selected"];
    [dict setValue:[NSNumber numberWithFloat:_arm] forKey:@"arm"];
    [dict setValue:[NSNumber numberWithFloat:_armLat] forKey:@"armLat"];
    
  

    return dict;
}

- (NSString*)convertToString {
    NSString* string = [NSString stringWithFormat:@"%@, %f, %@, %f, %f, %f, %f\n",
                        _station, self.weight, _weightInKg ? @"kg" : @"lbs",
                        self.arm, self.moment, self.armLat, self.momentLat];
    if (!_selected) {
        string = [@"-" stringByAppendingString:string];
    }
    return string;
}

- (void)setMoment:(float)moment {
    if (!_weightInKg) {
        moment = moment * 25.4 / 2.2046;
    }
    if (_weight) {
        _arm = moment/_weight;
    }
}

- (void)setMomentLat:(float)momentLat {
    if (!_weightInKg) {
        momentLat = momentLat * 25.4 / 2.2046;
    }
    if (_weight) {
        _armLat = momentLat/_weight;
    }
}

- (void)setArm:(float)arm {
    if (_weightInKg)
        _arm = arm;
    else
        _arm = arm * 25.4;
}

- (void)setArmLat:(float)armLat {
    if (_weightInKg)
        _armLat = armLat;
    else
        _armLat = armLat * 25.4;
}

- (void)setWeight:(float)weight {
    
    
    if (_weightInKg){
        _weight = weight;
        
        
    
    }
    else{
        
              _weight = weight / 2.2046;}
    
    
    
}

- (float)weight {
    
    _weightInKg = 1; // Claus
    
    if (_weightInKg) return _weight;
    return _weight * 2.2046;
}

- (float)arm {
    if (_weightInKg) return _arm;
    return _arm / 25.4;
}

- (float)armLat {
    if (_weightInKg) return _armLat;
    return _armLat / 25.4;
}


- (float)moment {
    return self.arm * self.weight;
}

- (float)momentLat {
    return self.armLat * self.weight;
}

@end

@implementation MedicalPackage {
    NSMutableArray* items_;
}

@synthesize items=items_;

- (id)init {
    self = [super init];
    if (self) {
        items_ = [NSMutableArray array];
        MedicalItem* item = [[MedicalItem alloc] init];
        item.selected = YES;
        [items_ addObject:item];
        [self addObserverForItem:item];
    }
    return self;
}

- (id)initWithSerializeKey:(NSString *)serializeKey {
    self = [super init];
    if (self) {
        _serializeKey = serializeKey;
        NSDictionary* dict = [[NSUserDefaults standardUserDefaults] valueForKey:serializeKey];
        _packageName = [dict valueForKey:@"packageName"];
        _longitudinal = [dict valueForKey:@"longitudinal"];
        _lateral = [dict valueForKey:@"lateral"];
        _selected = [[dict valueForKeyPath:@"selected"] boolValue];
        NSArray* items = [dict valueForKey:@"items"];
        items_ = [NSMutableArray array];
        if (items) {
            for (NSDictionary* itemDict in items) {
                MedicalItem* item = [[MedicalItem alloc] initWithDictionary:itemDict];
                [items_ addObject:item];
            }
        } else {
            MedicalItem* item = [[MedicalItem alloc] init];
            item.selected = YES;
            [items_ addObject:item];
        }
        [self addObservers];
        if (!dict) {
            [self serialize];
        }
    }
    return self;
}

+ (MedicalPackage*)emptyPackage {
    MedicalPackage* package = [[MedicalPackage alloc] init];
    package.items = nil;
    return package;
}

- (NSDictionary*)convertToDictionary {
    NSMutableDictionary* dict = [NSMutableDictionary dictionary];
    NSMutableArray* itemsArray = [NSMutableArray array];
    for (MedicalItem* item in items_) {
        [itemsArray addObject:[item convertToDictionary]];
    }
    [dict setValue:itemsArray forKey:@"items"];
    if (_packageName) [dict setValue:_packageName forKey:@"packageName"];
    if (_longitudinal) [dict setValue:_longitudinal forKey:@"longitudinal"];
    if (_lateral) [dict setValue:_lateral forKey:@"lateral"];
    [dict setValue:[NSNumber numberWithBool:_selected] forKeyPath:@"selected"];
    return dict;
}

- (NSString*)convertToString {
    NSString* string = [NSString stringWithFormat:@"Package=%@\n", _packageName];
    if (!_selected) {
        string = [@"-" stringByAppendingString:string];
    }
    for (MedicalItem* item in items_) {
        string = [string stringByAppendingString:[item convertToString]];
    }
    return string;
}

- (void)serialize {
    [[NSUserDefaults standardUserDefaults] setValue:[self convertToDictionary] forKey:_serializeKey];
}

- (void)wipeOut {
    [[NSUserDefaults standardUserDefaults] removeObjectForKey:_serializeKey];
}

- (void)setWeightInKg:(bool)weightInKg {
    _weightInKg = weightInKg;
    for (MedicalItem* item in items_) {
        item.weightInKg = _weightInKg;
    }
}


- (void)deleteItem:(MedicalItem *)item {
    [self removeObserverForItem:item];
    [items_ removeObject:item];
    [self serialize];
}

- (void)addItem:(MedicalItem *)item {
    [items_ addObject:item];
    item.weightInKg = _weightInKg;
    [self addObserverForItem:item];
    [self serialize];
}

- (float)totalWeight {
    float total = 0;
    for (MedicalItem* item in items_) {
        if (item.selected) total += item.weight;
    }
    return total * _selected;
}

- (float)totalMoment {
    float total = 0;
    for (MedicalItem* item in items_) {
        if (item.selected) total += item.moment;
    }
    return total * _selected;
}

- (float)totalMomentLat {
    float total = 0;
    for (MedicalItem* item in items_) {
        if (item.selected) total += item.momentLat;
    }
    return total * _selected;
}

- (float)totalArm {
    if (self.totalWeight) return self.totalMoment/self.totalWeight * _selected;
    return 0;
}

- (float)totalArmLat {
    if (self.totalWeight) return self.totalMomentLat/self.totalWeight * _selected;
    return 0;
}

- (void)setSelected:(bool)selected {
    _selected = selected;
    self.totalWeight = self.totalWeight; // Trigger observer for totalWeight property
    self.totalMoment = self.totalMoment; // Trigger observer
    self.totalMomentLat = self.totalMomentLat; // Trigger observer
    self.totalArm = self.totalArm; // Trigger observer
    self.totalArmLat = self.totalArmLat; // Trigger observer
    
    
}

- (void)addObserverForItem:(MedicalItem*)item {
    [item addObserver:self forKeyPath:@"weight" options:NSKeyValueObservingOptionNew context:nil];
    [item addObserver:self forKeyPath:@"selected" options:NSKeyValueObservingOptionNew context:nil];
    [item addObserver:self forKeyPath:@"station" options:NSKeyValueObservingOptionNew context:nil];
    [item addObserver:self forKeyPath:@"arm" options:NSKeyValueObservingOptionNew context:nil];
    [item addObserver:self forKeyPath:@"armLat" options:NSKeyValueObservingOptionNew context:nil];
    [item addObserver:self forKeyPath:@"moment" options:NSKeyValueObservingOptionNew context:nil];
    [item addObserver:self forKeyPath:@"momentLat" options:NSKeyValueObservingOptionNew context:nil];
}

- (void)removeObserverForItem:(MedicalItem*)item {
    [item removeObserver:self forKeyPath:@"weight"];
    [item removeObserver:self forKeyPath:@"selected"];
    [item removeObserver:self forKeyPath:@"station"];
    [item removeObserver:self forKeyPath:@"arm"];
    [item removeObserver:self forKeyPath:@"armLat"];
    [item removeObserver:self forKeyPath:@"moment"];
    [item removeObserver:self forKeyPath:@"momentLat"];
}

- (void)removeObservers {
    for (MedicalItem* item in items_) {
        [self removeObserverForItem:item];
    }
}

- (void)addObservers {
    for (MedicalItem* item in items_) {
        [self addObserverForItem:item];
    }
}

- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context {
    self.totalWeight = self.totalWeight; // Trigger observer for totalWeight property
    self.totalMoment = self.totalMoment; // Trigger observer
    self.totalMomentLat = self.totalMomentLat; // Trigger observer
    self.totalArm = self.totalArm; // Trigger observer
    self.totalArmLat = self.totalArmLat; // Trigger observer 
    [self serialize];
}

- (void)dealloc {
    [self removeObservers];
//    [[NSUserDefaults standardUserDefaults] removeObjectForKey:[self serializeKey]];
}

@end
